using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using System.Collections;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using BLOODUNITSHIPMENT = gov.va.med.vbecs.Common.VbecsTables.BloodUnitShipment;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using BLOODUNIT = gov.va.med.vbecs.Common.VbecsTables.BloodUnit;

namespace gov.va.med.vbecs.DAL
{
	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Carrie Van Stedum</Developer>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/13/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Implements data persistence logic for blood units. 
	///	Allows read and modify blood unit data in the VBECS database. 
	///</summary>	
	public class BloodUnit
	{
		#region Static Methods


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7473"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>DataTable with RowVersion</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7474"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>DataTable without RowVersion</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// spInsertBloodUnit, spUpdateBloodUnit
		/// </summary>
		/// <param name="isUpdate"></param>
		/// <returns></returns>
		public static DataTable GetEmptyBloodUnitTableSchema(bool isUpdate)
		{
			DataTable dtBloodUnit = new DataTable(BLOODUNIT.TableName);

			dtBloodUnit.Columns.Add(BLOODUNIT.BloodUnitGuid, typeof(System.Guid));
			dtBloodUnit.Columns.Add(BLOODUNIT.BloodProductGuid, typeof(System.Guid));
			dtBloodUnit.Columns.Add(BLOODUNIT.UnitProductCode, typeof(string));
			dtBloodUnit.Columns.Add(BLOODUNIT.DonationTypeId, typeof(int));
			dtBloodUnit.Columns.Add(BLOODUNIT.OriginalVolume, typeof(decimal));
			dtBloodUnit.Columns.Add(BLOODUNIT.DiscardedPlasmaVolume, typeof(decimal));
			dtBloodUnit.Columns.Add(BLOODUNIT.RecordStatusCode, typeof(string));

			if (isUpdate)
			{
				dtBloodUnit.Columns.Add(BLOODUNIT.RowVersion, typeof(byte[]));
			}
			else
			{
				dtBloodUnit.Columns.Add(BLOODUNIT.LabelUnitId, typeof(string));
				dtBloodUnit.Columns.Add(BLOODUNIT.EyeReadableUnitId, typeof(string));
				dtBloodUnit.Columns.Add(BLOODUNIT.UnitFdaRegistrationNumber, typeof(string));
				dtBloodUnit.Columns.Add(BLOODUNIT.UnitContainerLotNumber, typeof(string));
				dtBloodUnit.Columns.Add(BLOODUNIT.DivisionCode, typeof(string));
				dtBloodUnit.Columns.Add(BLOODUNIT.LabelTypeCode, typeof(string));
				dtBloodUnit.Columns.Add(BLOODUNIT.UnitDivisionId, typeof(string));
				dtBloodUnit.Columns.Add(BLOODUNIT.ProcessingTechId, typeof(string));
				dtBloodUnit.Columns.Add(BLOODUNIT.EntryMethodCode, typeof(string));
				dtBloodUnit.Columns.Add(BLOODUNIT.UnitInDate, typeof(System.DateTime));
			}

			return(dtBloodUnit);
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2002</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="426"> 
		///		<ExpectedInput>A valid product guid and unit id</ExpectedInput>
		///		<ExpectedOutput>A data table with one row</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="427"> 
		///		<ExpectedInput>Invalid UnitId/Product guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///	<summary>
		/// Retrieve blood unit details for a worklist builder control. 
		/// Altough all three parameters are optional at least one of them must be specified. 
		/// For more information on allowed parameters combinations see 
		/// 'sp GetBloodUnitWrkLst' stored procedure implementation.
		/// </summary>
		/// <param name="unitId">
		///		Blood unit ID. May be either eye readable ID or label ID. 
		///		Optional - may be null or blank.
		/// </param>
		/// <param name="bloodProductGuid">Product code. Optional - may be null or blank.</param>
		/// <returns>DataTable containing blood unit details</returns>
		public static DataTable GetBloodUnitDetails(string unitId, System.Guid bloodProductGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitWrkLst.unitid, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitWrkLst.productguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetBloodUnitWrkLst.divisioncode, SqlDbType.VarChar),
			};

			prms[0].Value = unitId;
			prms[1].Value = bloodProductGuid;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			return StoredProcedure.GetData(STOREDPROC.GetBloodUnitWrkLst.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/21/2005</CreationDate>
		///<Case type="0" testid ="8353"> 
		///		<ExpectedInput>A valid product guid, unit id and product code</ExpectedInput>
		///		<ExpectedOutput>A data table with one row</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8354"> 
		///		<ExpectedInput>Invalid UnitId/Product guid/product code</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		/// <summary>
		/// Retrieve blood unit details for a worklist builder control. 
		/// Altough all three parameters are optional at least one of them must be specified. 
		/// For more information on allowed parameters combinations see 
		/// 'sp GetBloodUnitWrkLst' stored procedure implementation.
		/// </summary>
		/// <param name="unitId" type="string">
		///   Blood unit ID. May be either eye readable ID or label ID. 
		///		Optional - may be null or blank.
		/// </param>
		/// <param name="bloodProductGuid" type="System.Guid">
		/// </param>
		/// <param name="productCode" type="string"></param>
		/// <returns>
		///     A System.Data.DataTable value...
		/// </returns>
		public static DataTable GetBloodUnitDetails(string unitId, System.Guid bloodProductGuid,string productCode)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitWrkLst.unitid, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitWrkLst.productguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetBloodUnitWrkLst.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitWrkLst.unitproductcode, SqlDbType.VarChar)
				};

			prms[0].Value = unitId;
			prms[1].Value = bloodProductGuid;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[3].Value = productCode;

			return StoredProcedure.GetData(STOREDPROC.GetBloodUnitWrkLst.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>03/19/2007</CreationDate>
		///
		///<Case type="0" testid ="8475"> 
		///		<ExpectedInput>A valid product guid, unit id and product code</ExpectedInput>
		///		<ExpectedOutput>A data table with one row</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8476"> 
		///		<ExpectedInput>Invalid UnitId/Product guid/product code</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		/// <summary>
		/// Retrieves blood unit data from the blood unit shipment table to see if a unit ever existed in our system and has since been modified
		/// DR 2610
		/// </summary>
		/// <param name="unitId"></param>
		/// <param name="bloodProductGuid"></param>
		/// <param name="productCode"></param>
		/// <returns></returns>
		public static DataTable GetIncomingShipmentBloodUnitDetails(string unitId, System.Guid bloodProductGuid,string productCode)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetBloodUnitDetailsFromBloodUnitShipment.unitid, SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetBloodUnitDetailsFromBloodUnitShipment.bloodproductguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitDetailsFromBloodUnitShipment.divisioncode, SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetBloodUnitDetailsFromBloodUnitShipment.fullproductcode, SqlDbType.VarChar)
			};

			prms[0].Value = unitId;
			prms[1].Value = bloodProductGuid;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[3].Value = productCode;

			return StoredProcedure.GetData(STOREDPROC.GetBloodUnitDetailsFromBloodUnitShipment.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/03/2003</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="832"> 
		///		<ExpectedInput>UnitId, Productcode</ExpectedInput>
		///		<ExpectedOutput>DataTable with Unit Infomration</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="833">
		///		<ExpectedInput>Invalid UnitID, ProductCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieves a single blood unit data based on eye readable blood unit id and product code.
		/// </summary>		
		/// <param name="eyeReadableUnitId">Eye readable blood unit ID</param>
		/// <param name="productCode">productCode</param>
		/// <param name="bloodProductGuid"></param>
		/// <returns>DataTable containing blood unit data</returns>
		public static DataTable GetBloodUnitByUnitIdProduct(string eyeReadableUnitId, string productCode, System.Guid bloodProductGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitByUnitIdProduct.eyereadableunitid, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitByUnitIdProduct.unitproductcode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitByUnitIdProduct.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitByUnitIdProduct.productguid, SqlDbType.UniqueIdentifier)
				};

			prms[0].Value = eyeReadableUnitId;
			prms[1].Value = productCode;
			prms[2].Value = LogonUser.LogonUserDivisionCode;
			prms[3].Value = bloodProductGuid;

			return StoredProcedure.GetData(STOREDPROC.GetBloodUnitByUnitIdProduct.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/23/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="834"> 
		///		<ExpectedInput>Divisoncode, Boolean</ExpectedInput>
		///		<ExpectedOutput>DataTable with Test Units</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="835">
		///		<ExpectedInput>Invalid DivisionCode, bool</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieves list of blood units that may be phenotyped in a given division.
		/// </summary>
		/// <param name="divisionCode">Division code</param>
		/// <param name="weakDTesting">Weak D Testing indicator</param>
		/// <returns>DataTable containing list of BloodUnits that may be phenotyped in a specified division</returns>
		public static DataTable GetPhenotypeUnits(string divisionCode, bool weakDTesting)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetPhenotypeUnits.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetPhenotypeUnits.weakdtesting, SqlDbType.Bit)
				};
			prms[0].Value = divisionCode;
			prms[1].Value = weakDTesting;

			return StoredProcedure.GetData(STOREDPROC.GetPhenotypeUnits.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/03/2003</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="836"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable with Unit Information</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="837">
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieves a single blood unit data based on blood unit GUID.
		/// </summary>
		/// <param name="bloodUnitGuid">Blood unit GUID</param>
		/// <returns>DataTable containing blood unit data</returns>
		public static DataTable GetBloodUnitByGuid(Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitByGuid.bloodunitguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetBloodUnitByGuid.divisioncode, SqlDbType.VarChar)
				};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return StoredProcedure.GetData(STOREDPROC.GetBloodUnitByGuid.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Mike      </Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="639"> 
		///		<ExpectedInput>Shipment identifier</ExpectedInput>
		///		<ExpectedOutput>The data table containing limited blood units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1890"> 
		///		<ExpectedInput>Invalid ShipmentGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieves all limited blood units in a given shipment from the database.
		/// Shipment GUID column is added to the resulting DataTable and its value is set for the first row.
		/// </summary>
		/// <param name="shipmentGuid">Shipment GUID</param>
		/// <param name="allowNegativeRh">If user turns off AntiD testing in UC2, they shouldn't be allowed to select negatvie units - else violate BR2.07</param>
		/// <param name="allowAllUnitStati">Allow Unit selection of units NOT in Limited Status, per BR2.42</param>
		/// <returns>DataTable containing list of all limited blood units in a given shipment</returns>
		public static DataTable GetLimitedBloodUnit(Guid shipmentGuid, bool allowNegativeRh, bool allowAllUnitStati)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetLimitedBloodUnit.shipmentguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetLimitedBloodUnit.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetLimitedBloodUnit.allownegativerh, SqlDbType.Bit),
					new SqlParameter(STOREDPROC.GetLimitedBloodUnit.allowallunitstati, SqlDbType.Bit)
				};

			if (shipmentGuid == System.Guid.Empty)
				prms[0].Value = DBNull.Value;
			else
				prms[0].Value = shipmentGuid;

			prms[1].Value = LogonUser.LogonUserDivisionCode;
			prms[2].Value = allowNegativeRh;
			prms[3].Value = allowAllUnitStati;

			DataTable result = StoredProcedure.GetData(STOREDPROC.GetLimitedBloodUnit.StoredProcName, prms).Tables[0];
			
			return result;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/03/2003</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="838"> 
		///		<ExpectedInput>DivisionCode, ProductTypeCodes</ExpectedInput>
		///		<ExpectedOutput>DataTable of Units</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="839">
		///		<ExpectedInput>Invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetLimitedBloodUnitsByDivision
		/// </summary>
		/// <param name="divisionCode">divisionCode</param>
		/// <param name="allowAllProductTypes">If true, user is not restricted to RBC types</param>
		/// <param name="allowNegativeRh">If user turns off AntiD testing in UC2, they shouldn't be allowed to select negatvie units - else violate BR2.07</param>
		/// <returns></returns>
		public static DataTable GetLimitedBloodUnitsByDivision(string divisionCode, bool allowAllProductTypes, bool allowNegativeRh)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetLimitedBloodUnitsByDivision.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetLimitedBloodUnitsByDivision.allowallproducttypes, SqlDbType.Bit),
					new SqlParameter(STOREDPROC.GetLimitedBloodUnitsByDivision.allownegativerh, SqlDbType.Bit)
				};

			prms[0].Value = divisionCode;
			prms[1].Value = allowAllProductTypes;
			prms[2].Value = allowNegativeRh;

			return StoredProcedure.GetData(STOREDPROC.GetLimitedBloodUnitsByDivision.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/14/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1808"> 
		///		<ExpectedInput>ShipmentGuid</ExpectedInput>
		///		<ExpectedOutput>Valid bloodunits datatable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1809"> 
		///		<ExpectedInput>Invalid ShipmentGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable Without ShipmentGuid column</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Fetches from database a list of all blood units in a given shipment that has arrive 
		/// in the past 5 days. Shipment GUID column is added to the resulting DataTable and 
		/// its value is set for the first row.
		/// If there are no units satisfying a search criteria 
		/// available - DataAccessLayerException is thrown.
		/// </summary>
		/// <param name="shipmentGuid">Shipment GUID</param>
		/// <returns>
		///		DataTable with list of all blood units in a given shipment 
		///		that has arrive in the past 5 days
		///	</returns>
		public static DataTable GetBloodUnits(Guid shipmentGuid)
		{
			DataTable result;

			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnits.shipmentguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetBloodUnits.divisioncode, SqlDbType.VarChar)
				};

			prms[0].Value = shipmentGuid;
			prms[1].Value = LogonUser.LogonUserDivisionCode;

			result = StoredProcedure.GetData(STOREDPROC.GetBloodUnits.StoredProcName, prms).Tables[0];
			result.Columns.Add(TABLE.Shipment.ShipmentGuid, typeof(System.Guid));

			if (result.Rows.Count != 0)
			{
				result.Rows[0][TABLE.Shipment.ShipmentGuid] = shipmentGuid;
			}
			return result;
		}

		/// <summary>
		/// Used to get static picture of units that were on an incoming shipment
		/// </summary>
		/// <param name="shipmentGuid"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitsIncomingShipment(Guid shipmentGuid)
		{
			DataTable result;

			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitsFromBloodUnitShipment.incomingshipmentguid, SqlDbType.UniqueIdentifier)
				};

			prms[0].Value = shipmentGuid;
			

			result = StoredProcedure.GetData(STOREDPROC.GetBloodUnitsFromBloodUnitShipment.StoredProcName, prms).Tables[0];
			result.Columns.Add(TABLE.Shipment.ShipmentGuid, typeof(System.Guid));

			if (result.Rows.Count != 0)
			{
				result.Rows[0][TABLE.Shipment.ShipmentGuid] = shipmentGuid;
			}
			return result;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/03/2003</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="840"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>DataTable of Units</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="841">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieves a list of blood units for editing in UC70
		/// </summary>
		/// <param name="previouslyAssociatedUnits">Indicates whether to retrieve units that have a historical patient association</param>
		/// <returns>DataTable with list of all blood units</returns>
		public static DataTable GetBloodUnitsForEdit(Boolean previouslyAssociatedUnits)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitsForEdit.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitsForEdit.previouslyassociatedunits, SqlDbType.Bit)
				};

			prms[0].Value = LogonUser.LogonUserDivisionCode;
			prms[1].Value = previouslyAssociatedUnits;

			return StoredProcedure.GetData(STOREDPROC.GetBloodUnitsForEdit.StoredProcName, prms).Tables[0];

		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/03/2003</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="842"> 
		///		<ExpectedInput>UnitId, ProductCode, DivisionCode, UnitDivisionId</ExpectedInput>
		///		<ExpectedOutput>BloodUnitGuid</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="843">
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>DataException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets a blood unit GUID based on unit label ID, unit product code and division code.
		/// Throws an exception if unit was not found.
		/// </summary>
		/// <param name="unitId">Blood unit ID - either an eye readable unit ID or label unit ID</param>
		/// <param name="unitProductCode">Unit product code</param>
		/// <param name="divisionCode">Division code</param>
		/// <param name="unitDivisionId">Unit Division ID - needed to check for alloQUATS</param>
		/// <returns>Blood unit GUID</returns>
		public static Guid GetBloodUnitGuid(string unitId, string unitProductCode, string divisionCode, string unitDivisionId)
		{
			if (unitId == null || unitId.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("unitId").ResString);
			}
			//
			if (unitProductCode == null || unitProductCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("unitProductCode").ResString);
			}
			//
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			//
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitGuid.labelunitid, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitGuid.productcode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitGuid.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitGuid.unitdivisionid, SqlDbType.VarChar)
				};
			//
			prms[0].Value = unitId;
			prms[1].Value = unitProductCode;
			prms[2].Value = divisionCode;
			//
			// This parameter is nullable, per the stored procedure, 
			// so we only pass it if it's valid.
			if (unitDivisionId != null && !unitDivisionId.Equals(string.Empty) && unitDivisionId.Length > 0)
			{
				prms[3].Value = unitDivisionId;
			}
			//
			DataTable dt = StoredProcedure.GetData(STOREDPROC.GetBloodUnitGuid.StoredProcName, prms).Tables[0];
			if (dt.Rows.Count <= 0)
			{
				throw(new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.UnitNotFound().ResString));
			}
			else
			{
				if (dt.Rows.Count > 1)
				{
					throw (new DataAccessLayerException("Possible Multiple Units with same Codabar product code.  Contact Administrator."));
				}
				else
				{
					return (Guid) dt.Rows[0][TABLE.BloodUnit.BloodUnitGuid];
				}
			}
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/08/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5788"> 
		///		<ExpectedInput>valid unitid, productcode, divisioncode, unitdivisionid, searchCriteria</ExpectedInput>
		///		<ExpectedOutput>BloodUnit DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5789"> 
		///		<ExpectedInput>Invalid unitid, productcode, divisioncode, unitdivisionid, searchCriteria</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodUnitsByIdAndProduct
		/// </summary>
		/// <param name="unitId">Blood unit ID - either an eye readable unit ID or label unit ID</param>
		/// <param name="unitProductCode">Unit product code</param>
		/// <param name="divisionCode">Division code</param>
		/// <param name="unitDivisionId">Unit Division ID - needed to check for alloQUATS; may be null</param>
		/// <param name="donationTypeCode">Donation Type COde - needed for ISBT only</param>
		/// <param name="searchCriteria"></param>
		/// <returns>Blood unit GUID</returns>
		public static DataTable GetBloodUnitsByIdAndProduct
			(
			string unitId, 
			string unitProductCode, 
			string divisionCode, 
			string unitDivisionId, 
			string donationTypeCode,
			Common.AdditionalBloodUnitSearchCriteria searchCriteria
			)
		{
			if (unitId == null || unitId.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("unitId").ResString);
			}
			//
			if (unitProductCode == null || unitProductCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("unitProductCode").ResString);
			}
			//
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}

			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitsByIdAndProduct.labelunitid, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitsByIdAndProduct.productcode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitsByIdAndProduct.divisioncode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitsByIdAndProduct.unitdivisionid, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitsByIdAndProduct.donationtypecode, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetBloodUnitsByIdAndProduct.searchcriteria, SqlDbType.Int)
				};

			prms[0].Value = unitId;
			prms[1].Value = unitProductCode;
			prms[2].Value = divisionCode;
			prms[3].Value = (unitDivisionId.Length == 0) ? null : unitDivisionId;
			prms[4].Value = (donationTypeCode.Length == 0) ? null : donationTypeCode;
			prms[5].Value = Common.Utility.GetAdditionalBloodUnitSearchCriteriaValueFromEnum(searchCriteria);
			//
			DataTable dt = StoredProcedure.GetData(STOREDPROC.GetBloodUnitsByIdAndProduct.StoredProcName, prms).Tables[0];

            //CR3294 - code removed that was throwing a DataAccessLayerException if Rows.Count <= 0
			return dt;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="995"> 
		///		<ExpectedInput>Blood type code</ExpectedInput>
		///		<ExpectedOutput>DataRow of rh factor for the blood type</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1892"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>IndexOfOutOfRangeException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets blood type and Rh factor information based on a blood type. BR_1.20, TT_1.02
		/// </summary>
		/// <param name="bloodTypeCode">Blood type code</param>
		/// <returns>DataRow containing blood type and Rh factor data for a given blood type</returns>
		public static DataRow GetBloodTypeRHFactorFromBloodTypeCode(string bloodTypeCode)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodTypeRHFactorFromBloodTypeCode.bloodtypecode, SqlDbType.VarChar)
				
				};

			prms[0].Value = bloodTypeCode;
			

			return StoredProcedure.GetData(STOREDPROC.GetBloodTypeRHFactorFromBloodTypeCode.StoredProcName, prms).Tables[0].Rows[0];
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7479"> 
		///		<ExpectedInput>DataSet, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7480"> 
		///		<ExpectedInput>DataSet UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     Save BloodUnit and all aggregates in the same transaction
		/// </summary>
		/// <param name="dsSave" type="System.Data.DataSet"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <returns>
		///     A bool value...
		/// </returns>
		public static bool SaveBloodUnitAndAggregates(System.Data.DataSet dsSave, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList sprocArray = new System.Collections.ArrayList();
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();


			//Loop through the dataset to include all the tables in transaction
			foreach (DataTable dtSave in dsSave.Tables)
			{
				if(dtSave.TableName == Common.VbecsTables.BloodUnit.TableName)
				{
					if(dtSave.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.PooledUnitsCount))
					{
						dtSave.Columns.Remove(Common.DatabaseConstants.ArtificialColumnNames.PooledUnitsCount);
					}
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId));
					sprocArray.Add(STOREDPROC.UpdateBloodUnit.StoredProcName);
				}

				else if (dtSave.TableName == Common.VbecsTables.BloodUnitMedia.TableName)
				{
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId));
					sprocArray.Add(STOREDPROC.UpdateBloodUnitMedia.StoredProcName);
				
				}
				else if (dtSave.TableName == Common.VbecsTables.BloodUnitFinance.TableName)
				{
					//SaveBloodUnitFinance doesn't want this parameter
					if (dtSave.Columns.Contains(BLOODUNITSHIPMENT.OutgoingShipmentGuid))
						dtSave.Columns.Remove(BLOODUNITSHIPMENT.OutgoingShipmentGuid);

					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId));
					sprocArray.Add(STOREDPROC.UpdateBloodUnitFinance.StoredProcName);
				}

				else if(dtSave.TableName == Common.VbecsTables.BloodUnitStatus.TableName)
				{
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId));
					sprocArray.Add(STOREDPROC.UpdateBloodUnitStatus.StoredProcName);
				}

				else if(dtSave.TableName == Common.VbecsTables.WorkloadEvent.TableName)
				{
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId));
					sprocArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
				}

				else if(dtSave.TableName == Common.VbecsTables.BloodUnitShipment.TableName)
				{
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId));
					sprocArray.Add(STOREDPROC.InsertBloodUnitShipment.StoredProcName);
				}
			}
			
			int returnValue = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);
			return (returnValue == 0);
		}


		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3776"> 
		///		<ExpectedInput>A set of data tables with valid data</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3777"> 
		///		<ExpectedInput>A set of data tables with invalid rowversions</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates certain fields from within UC 70, Edit Unit Information
		/// </summary>
		/// <param name="dtBloodUnit">Blood unit data table</param>
		/// <param name="dtBloodUnitMedia">Blood unit media data table</param>
		/// <param name="dtBloodUnitStatus">Blood unit status data table</param>
		/// <param name="dtInsertBloodUnitAntigens">Blood unit antigens data table</param>
		/// <param name="dtUpdateBloodUnitAntigens">Blood unit antigens data table</param>
		/// <param name="dtWorkload">Workload data, used for quarantine</param>
		/// <param name="bloodUnitTests">Blood unit tests data tables (for insert and update)</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="removeAllAntigens">Tells if we want to inactivate all antigens</param>
		/// <returns>True on success. Otherwise - false.</returns>
		public static bool Save
			(
			DataTable dtBloodUnit, 
			bool removeAllAntigens, 
			DataTable dtBloodUnitMedia, 
			DataTable dtBloodUnitStatus, 
			DataTable dtInsertBloodUnitAntigens, 
			DataTable dtUpdateBloodUnitAntigens, 
			DataTable dtWorkload,
			DataTable[] bloodUnitTests, 
			UpdateFunction lastUpdateFunctionId
			)
		{
			ArrayList alTables = new ArrayList();
			ArrayList alSprocs = new ArrayList();
			if (dtBloodUnit != null && dtBloodUnit.Rows.Count > 0)
			{
				Utility.RequireNonNullColumns(dtBloodUnit, STOREDPROC.UpdateBloodUnit.RequiredParameters);
				alTables.Add(Utility.AppendLastUpdateInformation(dtBloodUnit, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateBloodUnit.StoredProcName);
			}
			if (dtBloodUnitMedia != null && dtBloodUnitMedia.Rows.Count > 0)
			{
				Utility.RequireNonNullColumns(dtBloodUnitMedia, STOREDPROC.UpdateBloodUnitMedia.RequiredParameters);
				alTables.Add(Utility.AppendLastUpdateInformation(dtBloodUnitMedia, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateBloodUnitMedia.StoredProcName);
			}
			if (dtBloodUnitStatus != null && dtBloodUnitStatus.Rows.Count > 0)
			{
				Utility.RequireNonNullColumns(dtBloodUnitStatus, STOREDPROC.UpdateBloodUnitStatus.RequiredParameters);
				alTables.Add(Utility.AppendLastUpdateInformation(dtBloodUnitStatus, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateBloodUnitStatus.StoredProcName);
			}
			if (dtInsertBloodUnitAntigens != null && dtInsertBloodUnitAntigens.Rows.Count > 0)
			{
				Utility.RequireNonNullColumns(dtInsertBloodUnitAntigens, STOREDPROC.InsertBloodUnitAntigen.RequiredParameters);
				alTables.Add(Utility.AppendLastUpdateInformation(dtInsertBloodUnitAntigens, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.InsertBloodUnitAntigen.StoredProcName);
			}
			if (dtUpdateBloodUnitAntigens != null && dtUpdateBloodUnitAntigens.Rows.Count > 0)
			{
				Utility.RequireNonNullColumns(dtUpdateBloodUnitAntigens, STOREDPROC.UpdateBloodUnitAntigen.RequiredParameters);
				alTables.Add(Utility.AppendLastUpdateInformation(dtUpdateBloodUnitAntigens, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateBloodUnitAntigen.StoredProcName);
			}
			if (dtWorkload != null && dtWorkload.Rows.Count > 0)
			{
				Utility.RequireNonNullColumns(dtWorkload, STOREDPROC.InsertWorkloadEvents.RequiredParameters);
				alTables.Add(Utility.AppendLastUpdateInformation(dtWorkload, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
			}
			if (bloodUnitTests != null && bloodUnitTests.Length > 1)
			{
				if (bloodUnitTests[0].Rows.Count > 0)
				{
					Utility.RequireNonNullColumns(bloodUnitTests[0], STOREDPROC.InsertBloodUnitTest.RequiredParameters);
					alTables.Add(Utility.AppendLastUpdateInformation(bloodUnitTests[0], lastUpdateFunctionId));
					alSprocs.Add(STOREDPROC.InsertBloodUnitTest.StoredProcName);
				}
				if (bloodUnitTests[1].Rows.Count > 0)
				{
					Utility.RequireNonNullColumns(bloodUnitTests[1], STOREDPROC.UpdateBloodUnitTestResult.RequiredParameters);
					alTables.Add(Utility.AppendLastUpdateInformation(bloodUnitTests[1], lastUpdateFunctionId));
					alSprocs.Add(STOREDPROC.UpdateBloodUnitTestResult.StoredProcName);
				}
			}
			if(removeAllAntigens)
			{
				DataTable dtAntigen = new DataTable();
				dtAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.BloodUnitGuid, typeof(System.Guid));
				dtAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.AntigenTypeId, typeof(int));
				dtAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.DivisionCode, typeof(string));
				dtAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.Comments, typeof(string));
				dtAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.LastUpdateUser, typeof(string));
				dtAntigen.Columns.Add(Common.VbecsTables.BloodUnitAntigen.RowVersion, typeof(byte[]));
				//System.Data.DataTable dtBUAntigens = DAL.BloodUnitAntigen.GetBloodUnitAntigens((System.Guid) dtBloodUnit.Rows[0][BLOODUNIT.BloodUnitGuid], Common.LogonUser.LogonUserDivisionCode);
				System.Data.DataTable dtBUAntigens = DAL.BloodUnitAntigen.GetBloodUnitAntigens( ((System.Guid) dtBloodUnit.Rows[0][BLOODUNIT.BloodUnitGuid]).ToString(), Common.LogonUser.LogonUserDivisionCode);
				foreach(System.Data.DataRow dtRow in dtBUAntigens.Rows)
				{
					DataRow dtAntigenRow = dtAntigen.NewRow();
					dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.BloodUnitGuid] = dtBloodUnit.Rows[0][BLOODUNIT.BloodUnitGuid];
					dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.AntigenTypeId] = dtRow[TABLE.BloodUnitAntigen.AntigenTypeId];
					dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.Comments] = string.Empty;
					dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.LastUpdateUser] = Common.LogonUser.LogonUserName;
					dtAntigenRow[Common.VbecsTables.BloodUnitAntigen.RowVersion] = dtRow[TABLE.BloodUnitAntigen.RowVersion];				
					dtAntigen.Rows.Add(dtAntigenRow);
				}
				Utility.RequireNonNullColumns(dtAntigen, STOREDPROC.InactivateBloodUnitAntigen.RequiredParameters);
				alTables.Add(Utility.AppendLastUpdateInformation(dtAntigen, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.InactivateBloodUnitAntigen.StoredProcName);
			}
			//
			return (((new StoredProcedure()).TransactionalGetValue(alSprocs, alTables)) == 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/10/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3720"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table containing all units in a final status within user's division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3721"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function returns all the blood units that are in a final status within the user's division
		/// </summary>
		/// <returns>Blood unit datatable</returns>
		public static DataTable GetBloodUnitsInFinalStatus()
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitsInFinalStatus.divisioncode, SqlDbType.VarChar)
				};

			prms[0].Value = LogonUser.LogonUserDivisionCode;

			DataSet ds = StoredProcedure.GetData(STOREDPROC.GetBloodUnitsInFinalStatus.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3732"> 
		///		<ExpectedInput>Valid blood unit status and blood unit finance tables</ExpectedInput>
		///		<ExpectedOutput>Success indicator of true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3733"> 
		///		<ExpectedInput>Empty blood unit status and blood unit finance tables</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function removes a unit from a final status and udpdates the retrurn credit information
		/// </summary>
		/// <param name="bloodUnitStatus">Blood Unit Status data table</param>
		/// <param name="dtBloodUnitMediaRestoreRestiction">Data used to restore the previous restriction data for the unit</param>
		/// <param name="bloodUnitFinance">Blood Unit Finance data table</param>
		/// <param name="lastUpdateFunctionId">Last Update Function ID</param>
		///<param name="workloadEvents">Workloadevent data</param>
		///<param name="patientTransfusion">Patient Transfusion data table</param>
		///<param name="bloodUnitShipment">Blood Unit Shipment data table</param>
		///<param name="bloodUnitModification">Blood Unit Modification data table</param>
		///<param name="exceptionReport">Remove Unit Status exception report data table</param>
		///<param name="dtBCEMessages">BCE Messages</param>
		/// <returns>Success indicator</returns>
		public static bool RemoveFinalStatus(DataTable bloodUnitStatus, DataTable dtBloodUnitMediaRestoreRestiction, DataTable bloodUnitFinance, UpdateFunction lastUpdateFunctionId,
			DataTable workloadEvents, DataTable patientTransfusion, DataTable bloodUnitShipment, DataTable bloodUnitModification, DataTable exceptionReport, DataTable dtBCEMessages)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			if (lastUpdateFunctionId == UpdateFunction.Invalid)
			{
				throw new ArgumentException(StrRes.SysErrMsg.Common.InvalidFormat("last update function id").ResString);
			}
			//
			ArrayList alTables = new ArrayList();
			ArrayList alSprocs = new ArrayList();
			//
			if (bloodUnitStatus.Rows.Count > 0)
			{
				Common.Utility.RequireNonNullColumns(bloodUnitStatus, STOREDPROC.UpdateBloodUnitStatus.RequiredParameters);
				//
				alTables.Add(Utility.AppendLastUpdateInformation(bloodUnitStatus, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateBloodUnitStatus.StoredProcName);
				//CR 3180
				if(dtBloodUnitMediaRestoreRestiction != null)
				{
					if(dtBloodUnitMediaRestoreRestiction.Rows.Count >0)
					{
						alTables.Add(Utility.AppendLastUpdateInformation(dtBloodUnitMediaRestoreRestiction, lastUpdateFunctionId));
						alSprocs.Add(STOREDPROC.RestoreRestriction.StoredProcName);
					}
				}
				//
				if (bloodUnitFinance.Rows.Count > 0)
				{
					Common.Utility.RequireNonNullColumns(bloodUnitFinance, STOREDPROC.UpdateBloodUnitFinance.RequiredParameters);
					alTables.Add(Utility.AppendLastUpdateInformation(bloodUnitFinance, lastUpdateFunctionId));
					alSprocs.Add(STOREDPROC.UpdateBloodUnitFinance.StoredProcName);
				}
				if(workloadEvents.Rows.Count > 0)
				{
					alTables.Add(Utility.AppendLastUpdateInformation(workloadEvents, lastUpdateFunctionId));
					alSprocs.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
				}
				if (patientTransfusion.Rows.Count > 0)
				{
					alTables.Add(Utility.AppendLastUpdateInformation(patientTransfusion, lastUpdateFunctionId));
					alSprocs.Add(STOREDPROC.UpdatePatientTransfusionStatus.StoredProcName);
				}
				if (bloodUnitShipment.Rows.Count > 0)
				{
					alTables.Add(Utility.AppendLastUpdateInformation(bloodUnitShipment, lastUpdateFunctionId));
					alSprocs.Add(STOREDPROC.UpdateBloodUnitShipment.StoredProcName);
				}
				if (bloodUnitModification.Rows.Count > 0)
				{
					alTables.Add(Utility.AppendLastUpdateInformation(bloodUnitModification, lastUpdateFunctionId));
					alSprocs.Add(STOREDPROC.UpdateBloodUnitModificationStatus.StoredProcName);
				}
				alTables.Add(exceptionReport);
				alSprocs.Add(STOREDPROC.InsertExceptionRemoveFinalUnitStatus.StoredProcName);
			
				if (dtBCEMessages != null)
				{
					if (dtBCEMessages.Rows.Count > 0)
					{
						alTables.Add(dtBCEMessages);
						alSprocs.Add(STOREDPROC.HL7InsertMessageLog.StoredProcName);
					}
				}
			}
			//
			return (((new StoredProcedure()).TransactionalGetValue(alSprocs, alTables)) == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7481"> 
		///		<ExpectedInput>DataTables, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7482"> 
		///		<ExpectedInput>Invalid DataTables, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>RowVersion Exception</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RemoveUnitModification
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="bloodUnitModification"></param>
		/// <param name="bloodUnitMedia"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <param name="exceptionReport"></param>
		/// <param name="bloodUnitFinance">BloodUnitFinance table update - Unit Cost field mainly</param>
		/// <param name="dtWorkload">workload table</param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns></returns>
		public static bool RemoveUnitModification(DataTable bloodUnit, DataTable bloodUnitModification, DataTable bloodUnitMedia, UpdateFunction lastUpdateFunctionId, DataTable exceptionReport, DataTable bloodUnitFinance, DataTable dtWorkload, DataTable dtBCEMessages)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			ArrayList alTables = new ArrayList();
			ArrayList alSprocs = new ArrayList();
			//
			alTables.Add(Utility.AppendLastUpdateInformation(bloodUnitModification, lastUpdateFunctionId));
			alSprocs.Add(STOREDPROC.UpdateBloodUnitModificationStatus.StoredProcName);
			//
			if (bloodUnit.Rows.Count>0)
			{
				alTables.Add(Utility.AppendLastUpdateInformation(bloodUnit, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateBloodUnit.StoredProcName);
			}
			//
			if (bloodUnitMedia.Rows.Count>0)
			{
				alTables.Add(Utility.AppendLastUpdateInformation(bloodUnitMedia, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateBloodUnitMedia.StoredProcName);
			}
			//
			if (bloodUnitFinance.Rows.Count>0)
			{
				alTables.Add(Utility.AppendLastUpdateInformation(bloodUnitFinance, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateBloodUnitFinance.StoredProcName);
			}
			//
			if (dtWorkload.Rows.Count > 0)
			{
				alTables.Add(Utility.AppendLastUpdateInformation(dtWorkload, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
			}

			alTables.Add(exceptionReport);
			alSprocs.Add(STOREDPROC.InsertExceptionRemoveFinalUnitStatus.StoredProcName);
			//
			if (dtBCEMessages != null)
			{
				if (dtBCEMessages.Rows.Count > 0)
				{
					alTables.Add(dtBCEMessages);
					alSprocs.Add(STOREDPROC.HL7InsertMessageLog.StoredProcName);
				}
			}
			
			return (((new StoredProcedure()).TransactionalGetValue(alSprocs, alTables)) == 0);
		}

		///<Developers>
		///<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4030"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable w/row count > 0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4357"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Static function returns all the directed blood units that meet the requirements 
		/// of the following business rules: BR_71.01, BR_71.02 and BR_71.03
		/// </summary>
		/// <returns>Blood unit datatable</returns>
		public static DataTable GetBloodUnitsEligibleForCrossover()
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetBloodUnitsEligibleForCrossover.divisioncode, SqlDbType.VarChar)
				};

			prms[0].Value = LogonUser.LogonUserDivisionCode;

			DataSet ds = StoredProcedure.GetData(STOREDPROC.GetBloodUnitsEligibleForCrossover.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/01/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4031"> 
		///		<ExpectedInput>Valid bloodUnit and bloodUnitMedia DataTables</ExpectedInput>
		///		<ExpectedOutput>True (success)</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4032"> 
		///		<ExpectedInput>Empty bloodUnit DataTables</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function frees directed/restricted unit
		/// </summary>
		/// <param name="dtBloodUnit">Blood Unit data table</param>
		/// <param name="dtBloodUnitMedia">Blood Unit Media data table</param>
		/// <param name="dtWorkloadEvent">data for workload event</param>
		/// <param name="dtExceptionPatientRestriction">Exception data table</param>
		/// <param name="lastUpdateFunctionId">Last Update Function ID</param>
		/// <returns>Success indicator</returns>
		public static bool FreeDirectedUnit(DataTable dtBloodUnit, DataTable dtBloodUnitMedia, DataTable dtWorkloadEvent, DataTable dtExceptionPatientRestriction, UpdateFunction lastUpdateFunctionId)
		{
			Common.Utility.RequireNonNullColumns(dtBloodUnit, STOREDPROC.UpdateBloodUnit.RequiredParameters);
			Common.Utility.RequireNonNullColumns(dtBloodUnitMedia, STOREDPROC.UpdateBloodUnitMedia.RequiredParameters);
			Common.Utility.RequireNonNullColumns(dtWorkloadEvent, STOREDPROC.InsertWorkloadEvents.RequiredParameters);
			Common.Utility.RequireNonNullColumns(dtExceptionPatientRestriction, STOREDPROC.InsertExceptionPatientRestriction.RequiredParameters);
			//
			if (lastUpdateFunctionId == UpdateFunction.Invalid)
			{
				throw new ArgumentException(StrRes.SysErrMsg.Common.InvalidFormat("last update function id").ResString);
			}
			//
			ArrayList alTables = new ArrayList();
			ArrayList alSprocs = new ArrayList();
			//
			alTables.Add(Utility.AppendLastUpdateInformation(dtBloodUnit, lastUpdateFunctionId));
			alSprocs.Add(STOREDPROC.UpdateBloodUnit.StoredProcName);
			//
			alTables.Add(Utility.AppendLastUpdateInformation(dtBloodUnitMedia, lastUpdateFunctionId));
			alSprocs.Add(STOREDPROC.UpdateBloodUnitMedia.StoredProcName);
			//
			alTables.Add(dtWorkloadEvent);
			alSprocs.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
			//
			alTables.Add(dtExceptionPatientRestriction);
			alSprocs.Add(STOREDPROC.InsertExceptionPatientRestriction.StoredProcName);
			//
			bool test = (((new StoredProcedure()).TransactionalGetValue(alSprocs, alTables)) == 0);

			return test;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/30/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4052"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>Int - Number of Patients Assigned</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4358"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets count of all patients assigned to a blood unit.
		/// </summary>
		/// <param name="bloodUnitGuid">Blood unit GUID</param>
		/// <returns>count (int)</returns>
		public static int GetAssignedPatientCount(Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetBloodUnitAssignedPatientCount.BloodUnitGuid, SqlDbType.UniqueIdentifier)
			};
			//
			prms[0].Value = bloodUnitGuid;
			//
			return new StoredProcedure().GetIntReturnValue( STOREDPROC.GetBloodUnitAssignedPatientCount.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5455"> 
		///		<ExpectedInput>Valid dates</ExpectedInput>
		///		<ExpectedOutput>DataTable containing discarded unit data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5456"> 
		///		<ExpectedInput>Invalid dates</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of discarded blood units for a date range
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <returns></returns>
		public static DataTable GetDiscardedUnits(DateTime startDate, DateTime endDate) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetDiscardedUnits.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetDiscardedUnits.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetDiscardedUnits.divisioncode, System.Data.SqlDbType.VarChar,5)
			};
				
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;
				
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetDiscardedUnits.StoredProcName, prms);
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer> Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5880"> 
		///		<ExpectedInput>start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5881"> 
		///		<ExpectedInput>Invalid start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		/// <summary>
		/// For the selected date range and a division or all divisions, this method returns 
		///	blood units in discarded unit status for each component class and its subdivisions.  
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="isMultidivision"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetAdminReportBloodUnitsDiscarded(DateTime startDate, DateTime endDate, bool isMultidivision, string divisionCode) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsDiscarded.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsDiscarded.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsDiscarded.divisioncode, System.Data.SqlDbType.VarChar,5),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsDiscarded.ismultidivision, System.Data.SqlDbType.Bit)
			};
				
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			prms[2].Value = divisionCode;
			prms[3].Value = isMultidivision;
				
			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetAdminReportBloodUnitsDiscarded.StoredProcName, prms).Tables[0];
			dt.TableName = "BloodUnitsDiscarded";
			return dt;
		}

		///<Developers>
		///	<Developer> Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5878"> 
		///		<ExpectedInput>start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5879"> 
		///		<ExpectedInput>Invalid start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		/// <summary>
		/// For the selected date range and a division or all divisions, this method returns 
		///	blood units in modified unit status for each component class and its subdivisions
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="isMultidivision"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetAdminReportBloodUnitsModifiedPrepared(DateTime startDate, DateTime endDate, bool isMultidivision, string divisionCode) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsModifiedPrepared.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsModifiedPrepared.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsModifiedPrepared.divisioncode, System.Data.SqlDbType.VarChar,5),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsModifiedPrepared.ismultidivision, System.Data.SqlDbType.Bit)
			};
				
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			prms[2].Value = divisionCode;
			prms[3].Value = isMultidivision;
				
			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetAdminReportBloodUnitsModifiedPrepared.StoredProcName, prms).Tables[0];
			dt.TableName = "BloodUnitsModifiedPrepared";
			return dt;
		}

		///<Developers>
		///	<Developer> Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5876"> 
		///		<ExpectedInput>start date, bool</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5877"> 
		///		<ExpectedInput>Invalid start date, bool</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		/// <summary>
		/// For the selected date range and a division or all divisions, this method returns 
		///	expired blood units for each component class and its subdivisions.
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="isMultidivision"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetAdminReportBloodUnitsOutdated(DateTime startDate, bool isMultidivision, string divisionCode) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsOutdated.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsOutdated.divisioncode, System.Data.SqlDbType.VarChar,5),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsOutdated.ismultidivision, System.Data.SqlDbType.Bit)
			};
				
			prms[0].Value = startDate;
			prms[1].Value = divisionCode;
			prms[2].Value = isMultidivision;
				
			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetAdminReportBloodUnitsOutdated.StoredProcName, prms).Tables[0];
			dt.TableName = "BloodUnitsOutdated";
			return dt;
		}

		///<Developers>
		///	<Developer> Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5874"> 
		///		<ExpectedInput>start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5875"> 
		///		<ExpectedInput>Invalid start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		/// <summary>
		/// For the selected date range and a division or all divisions, this method returns 
		///	blood units received during incoming shipment for each component class and its subdivisions.  
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="isMultidivision"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetAdminReportBloodUnitsReceived(DateTime startDate, DateTime endDate, bool isMultidivision, string divisionCode) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsReceived.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsReceived.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsReceived.divisioncode, System.Data.SqlDbType.VarChar,5),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsReceived.ismultidivision, System.Data.SqlDbType.Bit)
			};
				
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			prms[2].Value = divisionCode;
			prms[3].Value = isMultidivision;

				
			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetAdminReportBloodUnitsReceived.StoredProcName, prms).Tables[0];
			dt.TableName = "BloodUnitsReceived";
			return dt;
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5882"> 
		///		<ExpectedInput>start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5883"> 
		///		<ExpectedInput>Invalid start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		/// <summary>
		/// For the selected date range and a division or all divisions, this method returns 
		///	blood units in transfered blood unit status for each component class and its subdivisions.  
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="isMultidivision"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetAdminReportBloodUnitsShipped(DateTime startDate, DateTime endDate, bool isMultidivision, string divisionCode) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsShipped.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsShipped.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsShipped.divisioncode, System.Data.SqlDbType.VarChar,5),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsShipped.ismultidivision, System.Data.SqlDbType.Bit)
			};
				
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			prms[2].Value = divisionCode;
			prms[3].Value = isMultidivision;

				
			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetAdminReportBloodUnitsShipped.StoredProcName, prms).Tables[0];
			dt.TableName = "BloodUnitsShipped";
			return dt;
		}
		
		///<Developers>
		///	<Developer> Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5872"> 
		///		<ExpectedInput>start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5873"> 
		///		<ExpectedInput>Invalid start date, end date, bool</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		/// <summary>
		/// For the selected date range and a division or all divisions, this method returns 
		///	blood units in transfused blood unit status for each component class and its subdivisions. 
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="isMultidivision"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetAdminReportBloodUnitsTransfused(DateTime startDate, DateTime endDate, bool isMultidivision, string divisionCode) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsTransfused.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsTransfused.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsTransfused.divisioncode, System.Data.SqlDbType.VarChar,5),
				new SqlParameter(STOREDPROC.GetAdminReportBloodUnitsTransfused.ismultidivision, System.Data.SqlDbType.Bit)
			};
				
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			prms[2].Value = divisionCode;
			prms[3].Value = isMultidivision;

				
			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetAdminReportBloodUnitsTransfused.StoredProcName, prms).Tables[0];
			dt.TableName = "BloodUnitsTransfused";
			return dt;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/3/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6823"> 
		///		<ExpectedInput>Valid Dates</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6824"> 
		///		<ExpectedInput>Invalid Dates</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of inactivated units (for Cost Accounting Report)
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <returns></returns>
		public static DataTable GetInactivatedUnitsList(DateTime startDate, DateTime endDate) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetInactivatedUnitsList.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetInactivatedUnitsList.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetInactivatedUnitsList.divisioncode, System.Data.SqlDbType.VarChar)
			};
				
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

				
			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetInactivatedUnitsList.StoredProcName, prms).Tables[0];
			dt.TableName = "InactivatedUnits";
			return dt;
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7483"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7484"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetModifiedBloodUnits
		/// </summary>
		/// <returns></returns>
		public static DataTable GetModifiedBloodUnits()
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetModifiedBloodUnits.divisioncode, SqlDbType.VarChar)
				};

			prms[0].Value = LogonUser.LogonUserDivisionCode;

			DataSet ds = StoredProcedure.GetData(STOREDPROC.GetModifiedBloodUnits.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9180"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9181"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets BloodUnitId and ProductCode in an array for a BloodUnitGuid
		/// </summary>
		public static string [] GetBloodUnitIdAndProductCodeForBloodUnitGuid(Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetBloodUnitIdAndProductCodeForBloodUnitGuid.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier)
			};
			//
			prms[0].Value = bloodUnitGuid;
			//
			DataRow drUnitData = Common.StoredProcedure.GetSingleDataRow(STOREDPROC.GetBloodUnitIdAndProductCodeForBloodUnitGuid.StoredProcName, prms);
			//
			if ( drUnitData != null )
			{
				return new string[2]{ drUnitData[TABLE.BloodUnit.EyeReadableUnitId].ToString(), drUnitData[TABLE.BloodUnit.UnitProductCode].ToString() };
			}
			//
			return null;
		}

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/14/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9237"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9238"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 3006
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static bool BloodUnitExists(Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.BloodUnitExists.BloodUnitGuid, SqlDbType.UniqueIdentifier)
			};
			//
			prms[0].Value = bloodUnitGuid;
			//
			return new Common.StoredProcedure().GetBoolReturnValue( STOREDPROC.BloodUnitExists.StoredProcName, prms );
		}

		#endregion
	}
}